/*
 * File: BankingDisplay.java
 * ======================================================
 * A display that visualizes the bank failures that have
 * occurred in the past decade.  Each state is colored
 * based on how many failures have occurred.
 */

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import acm.graphics.*;

public class BankingDisplay extends GCanvas implements ComponentListener {
	/* The viewpoint coordinates - the minimum and maximum longitude
	 * and latitude.
	 */
	private static final double MIN_LONGITUDE = -180;
	private static final double MAX_LONGITUDE = -60;
	
	private static final double MIN_LATITUDE = +15;
	private static final double MAX_LATITUDE = +75;
	
	/* A map from names of states to the Border object defining the border of
	 * those states.  This is used to reconstruct the state GCompounds after
	 * we resize the window.
	 */
	private Map<String, Border> borders;
	
	/* A map from the names of states to a GCompound of that state. */
	private Map<String, GCompound> states = new HashMap<String, GCompound>();
	
	/* A map from the name of a state to the total number of failures that have
	 * occurred in it.
	 */
	private Map<String, Integer> totals = new HashMap<String, Integer>();
	
	/**
	 * Constructs a BankingDisplay given a map associating state names with
	 * their borders.
	 * 
	 * @param borders The borders of each US region.
	 */
	public BankingDisplay(Map<String, Border> borders) {
		addComponentListener(this);
		this.borders = borders;
		
		setBackground(Color.BLACK);
		recomputeShapes();
	}
	
	/**
	 * Removes all information from the display, but does not force a redraw.
	 */
	public void clear() {
		totals.clear();
	}
	
	/**
	 * Adds the given bank failure to the display.
	 * 
	 * @param toAdd The failure to add.
	 */
	public void addFailure(BankFailure toAdd) {
		/* If we don't have any information about the state in which this failure
		 * occurred, mark that there's just one failure there.
		 */
		if (!totals.containsKey(toAdd.getState())) {
			totals.put(toAdd.getState(), 1);
		}
		/* Otherwise, increment the total by one. */
		else {
			totals.put(toAdd.getState(), totals.get(toAdd.getState()) + 1);
		}
	}
	
	/**
	 * Updates the display.
	 */
	public void update() {
		removeAll();
		drawStates();
		drawTotal();
	}
	
	/**
	 * Draws a label indicating how many total failures have occurred.
	 */
	private void drawTotal() {
		/* Count up how many failures there have been. */
		int totalFailures = 0;
		for (Integer failures: totals.values()) {
			totalFailures += failures;
		}
		
		/* Construct a GLabel with this information. */
		GLabel total = new GLabel("Total failures: " + totalFailures, 0, getHeight());
		total.setFont("DejaVuSerif-BOLD-24");
		total.setColor(Color.YELLOW);
		add(total);
	}
	
	/**
	 * Recomputes all of the GCompounds representing the boundaries of the
	 * states, given the new information about the graphics window.
	 */
	private void recomputeShapes() {
		/* Wipe out all our old information. */
		states.clear();
		
		/* Construct a GCompound for each state. */
		for (String state: borders.keySet()) {
			Border currBorder = borders.get(state);
			GCompound compound = new GCompound();
			
			/* Each border may have multiple shapes in it.  For each of those
			 * shapes, construct a polygon and add it to the compound.
			 */
			Iterator<Shape> shapeIter = currBorder.iterator();
			while (shapeIter.hasNext()) {
				compound.add(createPolygonForShape(shapeIter.next()));
			}
			
			/* Add the new compound to the map. */
			states.put(state, compound);
		}
	}
	
	/**
	 * Given a Shape, constructs a filled GPolygon whose outline is given by the
	 * points in the shape.
	 * 
	 * @param shape The shape for which we should create a polygon.
	 * @return The constructed polygon.
	 */
	private GPolygon createPolygonForShape(Shape shape) {
		GPolygon result = new GPolygon();
		result.setFilled(true);
	
		/* Visit each point, converting the longitude and latitude of the points
		 * into x and y coordinates on the display.
		 */
		Iterator<GPoint> pointIter = shape.iterator();
		while (pointIter.hasNext()) {
			GPoint currPoint = pointIter.next();
			double x = longitudeToXCoordinate(currPoint.getX());
			double y = latitudeToYCoordinate(currPoint.getY());
			result.addVertex(x, y);
		}
		return result;
	}
	
	/**
	 * Draws all of the US states, coloring each appropriately.
	 */
	private void drawStates() {
		for (String state: states.keySet()) {
			/* If we have records for this state, look up those records.
			 * Otherwise, there must be no records here.
			 */
			int total = totals.containsKey(state)? totals.get(state) : 0;
			
			/* Grab the compound and color it appropriately. */
			GCompound shape = states.get(state);
			shape.setColor(colorForTotal(total));
			
			add(shape);
		}
	}
	
	/**
	 * Computes the logistic function, which is used as a subroutine for 
	 * determining the color of each state.
	 * 
	 * @param x The input value
	 * @return logistic(x)
	 */
	private float logistic(double x) {
		return (float) (1.0 / (1.0 + Math.exp(-x)));
	}
	
	/**
	 * Given a total number of failures in a location, returns a color for that
	 * many failures.
	 * 
	 * This color is computed by interpolating between green and red according to
	 * a logistic function.
	 * 
	 * @param total The total number of failures.
	 * @return The color to use to represent that many failures.
	 */
	private Color colorForTotal(int total) {
		/* We want to range between green (hue 1/3) and red (hue 0).  To do this, we
		 * begin with green (1 / 3), then use the logistic function to do some
		 * interpolation.  Since the logistic will be between 0.5 and 1, we
		 * subtract out 0.5 and double it to get a value in the range 0 to 1.
		 * We then scale that to be between 0 and 1/3, and subtract it from the
		 * green component.
		 */
		float intensity = 1 / 3f - 2 * (logistic(total / 2.0) - 0.5f) / 3.0f;
		return new Color(Color.HSBtoRGB(intensity, 1f, 1f));
	}
	
	/**
	 * Given a raw longitude, returns the screen x coordinate where
	 * it should be displayed.
	 * 
	 * @param longitude The longitude in question.
	 * @return Where it maps to as an x coordinate.
	 */
	private double longitudeToXCoordinate(double longitude) {
		return getWidth() * (longitude - MIN_LONGITUDE) / (MAX_LONGITUDE - MIN_LONGITUDE); 
	}
	
	/**
	 * Given a raw latitude, returns the screen y coordinate where
	 * it should be displayed.
	 * 
	 * @param latitude The latitude in question.
	 * @return Where it maps to as a y coordinate.
	 */
	private double latitudeToYCoordinate(double latitude) {
		return getHeight() * (1.0 - (latitude - MIN_LATITUDE) / (MAX_LATITUDE - MIN_LATITUDE)); 
	}
	
	public void componentHidden(ComponentEvent arg) { }
	public void componentMoved(ComponentEvent arg) { }
	public void componentResized(ComponentEvent arg) { recomputeShapes(); update(); }
	public void componentShown(ComponentEvent arg) { }
}
